%% Bazilinskyy, P., Kooijman, L., Dodou, D., & De Winter, J. C. F. (2021). How should external Human-Machine Interfaces behave? Examining the effects of colour, position, message, activation distance, vehicle yielding, and visual distraction among 1,434 participants. Applied Ergonomics.
% MATLAB script built by Joost de Winter, Pavlo Bazilinksyy, & Lars Kooijman
% Tested in MATLAB R2021a

clear all;close all;clc; %#ok<*CLALL>
opengl hardware
%% Load data from the RT datases
savepng=1; % if set to 1 then save .png files
readraw=0; % if 1 then read entries_1.xlsx and entries_2.xlsx
if readraw==1 %% Read Data from RT files
     raw_kp =readmatrix('entries_1.xlsx','outputtype','string','NumHeaderLines',0); %#ok<NASGU> % Import Excel file with Keypress data
     reduce_data
     save('alldata1.mat','IDrt','RTrt','VNrt','ETrt'); % save matrices with participant ID, keypress time, and video number
     raw_kp =readmatrix('entries_2.xlsx','outputtype','string','NumHeaderLines',0); % Import Excel file with Keypress data
     reduce_data
     save('alldata2.mat','IDrt','RTrt','VNrt','ETrt'); % save matrices with participant ID, keypress time, and video number
 end
load alldata2.mat
IDrt2=IDrt;
RTrt2=RTrt;
VNrt2=VNrt;
ETrt2=ETrt;
load alldata1.mat
IDrt=[mean(IDrt,2,'omitnan');mean(IDrt2,2,'omitnan')]; % participant ID
RTrt=[RTrt NaN(size(RTrt,1),size(RTrt2,2)-size(RTrt,2));RTrt2]; % combine key press times of the two databases; remaining 'space' filled with NaNs
VNrt=[VNrt NaN(size(VNrt,1),size(VNrt2,2)-size(VNrt,2));VNrt2]; % combine video numbers of the two databases; remaining 'space' filled with NaNs
ETrt=[ETrt NaN(size(ETrt,1),size(ETrt2,2)-size(ETrt,2));ETrt2]; % combine elapsed times of the two databases; remaining 'space' filled with NaNs
clear IDrt2 RTrt2 VNrt2 ETrt2
%% Process the RT files into a matrix 'key_presses' containing all key presses per participant number and video number
key_presses=NaN(2400,276,500);
video_shown=zeros(2400,276);
AllVideoIndexes=NaN(length(IDrt),20);
for i1=1:length(IDrt) % loop over all rows in the database
    VideoIndexes=find(VNrt(i1,:)>0); % find indexes of starts of new videos
    ETIndexes=find(ETrt(i1,:)>0); % find indexes of 'elapsed times'
    participantnumber=IDrt(i1);
    
    DET=diff(ETrt(i1,ETIndexes)); % differences in elapsed time
    invalidblock=(sum(DET>12500)>=2)|sum(DET>13000); % Two or more trials longer than 12.5 s or one or more trials longer than 13 s (per block of 5). This is a signal that the video was not played back properly (slow graphics card of participant?)

    AllVideoIndexes(i1,1:length(VideoIndexes))=VideoIndexes;
    if ~isnan(participantnumber) && length(unique(VNrt(i1,VideoIndexes)))==5 && invalidblock==0 % there need to be 5 per batch
        for i2=1:length(VideoIndexes)
            videonumber=VNrt(i1,VideoIndexes(i2));
            video_shown(participantnumber,videonumber)=1;
            if i2==1
                rts_for_video=RTrt(i1,1:VideoIndexes(i2)-1);
            else
                rts_for_video=RTrt(i1,VideoIndexes(i2-1)+1:VideoIndexes(i2)-1);
            end
            rts_for_video=rts_for_video(rts_for_video>0); % remove NaNs
            if length(rts_for_video)>500 % limit the length (50 seconds) to save memory
                rts_for_video(500:end)=[];
            end                
            key_presses(participantnumber,videonumber,1:length(rts_for_video))=rts_for_video;
        end
    end
end
%% load FigureEight data and Video data
[~,~,raw_cf] = xlsread('survey_data.xlsx'); % Import Excel file with FigureEight data (crowd study)
raw_cf(1,:) = []; % Delete first row with headers

video_info = xlsread('video_info.xlsx','B2:I277'); % Import Excel file with video information
%% Process FigureEight data
disp(['Number of respondents in FigureEight = ' num2str(size(raw_cf,1))])
X=NaN(size(raw_cf,1),18);
temp=raw_cf(:,15);X(:,1)=strcmp(temp,'no')+2*strcmp(temp,'yes'); % Instructions understood
temp=raw_cf(:,33);X(:,2)=1*strcmp(temp,'female')+2*strcmp(temp,'male')-1*strcmp(temp,'i_prefer_not_to_respond'); % Gender
temp=raw_cf(:,32);for i=1:length(temp);try if strcmp(temp(i),'?');X(i,3)=NaN;else;X(i,3)= cell2mat(temp(i));end;catch error;X(i,3)=NaN;end;end % Age
X(X(:,3)>110,3)=NaN; % People who report age greater than 110 years
temp=raw_cf(:,14);for i=1:length(temp);try if strcmp(temp(i),'?');X(i,4)=NaN;else;X(i,4)= cell2mat(temp(i));end;catch error;X(i,4)=NaN;end;end % Age of obtaining driver's license
X(X(:,4)>110,4)=NaN; % People who report licence more than 110 years
temp=raw_cf(:,34);for i=1:length(temp);X(i,5)=1*strcmp(temp(i),'private_vehicle')+2*strcmp(temp(i),'public_transportation')+3*strcmp(temp(i),'motorcycle')+4*strcmp(temp(i),'walkingcycling')+5*strcmp(temp(i),'other')-1*strcmp(temp(i),'i_prefer_not_to_respond')-1*isnan(temp{i}(1));end % Primary mode of transportation
temp=raw_cf(:,29);for i=1:length(temp);X(i,6)=1*strcmp(temp(i),'never')+2*strcmp(temp(i),'less_than_once_a_month')+3*strcmp(temp(i),'once_a_month_to_once_a_week')+4*strcmp(temp(i),'1_to_3_days_a_week')+5*strcmp(temp(i),'4_to_6_days_a_week')+6*strcmp(temp(i),'every_day')-1*strcmp(temp(i),'i_prefer_not_to_respond')-1*isnan(temp{i}(1));end % How many times in past 12 months did you drive a vehicle
temp=raw_cf(:,13);for i=1:length(temp);try X(i,7)=1+cell2mat(temp(i));catch error;X(i,7)=1*strcmp(temp(i),'0_km__mi')+2*strcmp(temp(i),'1__1000_km_1__621_mi')+3*strcmp(temp(i),'1001__5000_km_622__3107_mi')+4*strcmp(temp(i),'5001__15000_km_3108__9321_mi')+5*strcmp(temp(i),'15001__20000_km_9322__12427_mi')+6*strcmp(temp(i),'20001__25000_km_12428__15534_mi')+7*strcmp(temp(i),'25001__35000_km_15535__21748_mi')+8*strcmp(temp(i),'35001__50000_km_21749__31069_mi')+9*strcmp(temp(i),'50001__100000_km_31070__62137_mi')+10*strcmp(temp(i),'more_than_100000_km_more_than_62137_mi')-1*strcmp(temp(i),'i_prefer_not_to_respond');end;end % Mileage
temp=raw_cf(:,17);for i=1:length(temp);try X(i,8)=1+cell2mat(temp(i));catch error;X(i,8)=7*strcmp(temp(i),'more_than_5')-1*strcmp(temp(i),'i_prefer_not_to_respond');end;end % Number of accidents
temp=raw_cf(:,9);Country=cell(size(X,1),1);for i=1:length(temp);try Country(i)=unique(temp(i));catch error;Country(i)={'NaN'};end;end % Country
% Driver behaviour questionnaire (DBQ)
temp=raw_cf(:,18:24);X(:,9:15)=1*strcmp(temp,'0_times_per_month')+2*strcmp(temp,'1_to_3_times_per_month')+3*strcmp(temp,'4_to_6_times_per_month')+4*strcmp(temp,'7_to_9_times_per_month')+5*strcmp(temp,'10_or_more_times_per_month')-1*strcmp(temp,'i_prefer_not_to_respond'); % DBQ violations
X(X<0)=NaN;
temp=raw_cf(:,29);Comments=cell(size(X,1),1);for i=1:length(temp);Comments(i)=temp(i);end % Comments
%% Study completion time
for i=1:size(raw_cf,1)
    X(i,17)=datenum(datetime(raw_cf{i,4},'Locale','fi_FI'));
    X(i,18)=datenum(datetime(raw_cf{i,2},'Locale','fi_FI'));
    X(i,16)=round(2400*36*(X(i,18)-X(i,17)));
end
%% Transform to key_presses to key_pressesf0, containing the key press data in 0.1-s increments
key_pressesf0 = NaN(size(key_presses,1),size(key_presses,2),200);
for i = 1 : size(key_presses, 1)
    for j = 1:size(key_presses, 2)
        if video_shown(i,j)==1
            temp = squeeze(key_presses(i,j,:))/1000;
            temp(isnan(temp)) = [];
            if max(temp)>12.5 % if there is more than 12.5 seconds of data, then the trial must be invalid
                temp=[];
            end
            
            if length(temp)>=2 % fill data gap for first half a second of holding the key 
                if temp(2)-temp(1)>0.475 && temp(2)-temp(1)<0.525
                    temp=[temp(1);transpose(temp(1):0.04:temp(2));temp(2:end)];
                end
            end
            
            O = 10 * unique(round(temp,1));  % Indices where button is pressed (in 0.1-second increments)
            key_pressesf0(i,j,:) = 0;
            O(O==0) = [];
            key_pressesf0(i,j,O) = 1;
        end
    end
end
%% Create new matrix key_pressesf, containing the RT data of valid FigureEight respondents only
IDcf=NaN(size(raw_cf,1),1);
key_pressesf1=NaN(size(raw_cf,1),size(key_pressesf0,2),size(key_pressesf0,3));
for i = 1 : size(raw_cf,1) % Loop across all lines of the FigureEight data
    temp    = raw_cf{i,58}; % cell with participant ID number FigureEight
    IDcf(i) = str2double(temp(regexp(temp,'UNITY')+5:end)); % Extract participant ID number FigureEight
    key_pressesf1(i,:,:) = key_pressesf0(IDcf(i),:,:);
end
invalid1=find(sum(~isnan(key_pressesf1(:,:,110)),2)<30); % Fewer than 30 videos
%% Find rows with identical IP addresses
y = NaN(size(X(:,1)));
IPCF=NaN(size(raw_cf,1),1);
for i=1:size(raw_cf,1)
    try IPCF(i)=str2double(strrep(raw_cf(i,12),'.',''));
    catch
        IPCF(i)=cell2mat(raw_cf(i,12));
    end
end % reduce IP addresses of FigureEight to a single number
for i=1:size(X,1)
    temp=find(IPCF==IPCF(i));
    if length(temp)==1 % if the IP address occurs only once
        y(i)=1; % keep
    elseif length(temp)>1 % if the IP addres occurs more than once
        y(temp(1))=1; % keep the first survey for that IP address
        y(temp(2:end))=2; % do not keep the other ones
    end
end
invalid5=find(y>1); % respondents who completed the survey more than once (i.e., remove the doublicates)
%% Remove participants who did not meet the criteria
invalid2 = find(X(:,1)==1); % respondents who did not read instructions
invalid3 = find(X(:,3)<18);  % respondents who indicated they are under 18 years old
invalid4 = find(X(:,16)<500);  % respondents who took less than 500 s to complete the study (unrealistically fast)
invalid = unique([invalid1;invalid2;invalid3;invalid4;invalid5]); % Combine invalid indexes

X_filtered = X; % copy data before filtering
Country_filtered = Country; % copy data before filtering
Comments_filtered = Comments; % copy data before filtering
raw_cf_filtered = raw_cf; % copy data before filterin
key_pressesf=key_pressesf1; % copy data before filterin

X_filtered(invalid,:)=[]; % Remove invalid respondents
Country_filtered(invalid)=[]; % Remove invalid respondents
Comments_filtered(invalid)=[]; % Remove invalid respondents
raw_cf_filtered(invalid,:)=[]; % Remove invalid respondents
key_pressesf(invalid,:,:)=[]; % Remove invalid respondents
%% Output study stats
disp(['Study completion time (after filtering) Mean = ' num2str(mean(X_filtered(:,16)/60,'omitnan'))]);
disp(['Study completion time (after filtering) SD = ' num2str(std(X_filtered(:,16)/60,'omitnan'))]);
disp(['Number of participants <40 videos = ' num2str(length(invalid1))]);
disp(['Number of respondents who did not read instructions = ' num2str(length(invalid2))])
disp(['Number of respondents under 18 = ' num2str(length(invalid3))])
disp(['Number of respondents that took less than 500 s = ' num2str(length(invalid4))])
disp(['Number of responses coming from the same IP = ' num2str(length(invalid5))])
disp(['Number of respondents removed = ' num2str(length(invalid))])
disp(['Number of respondents included in the analysis:  ' num2str(size(X_filtered,1))]);
disp(['Number of countries (after filtering):  ' num2str(length(unique(Country_filtered)))]);
disp(['Number of videos per person, min max, mean, SD, median : ' num2str((min(sum(~isnan(key_pressesf(:,:,110)),2)))) '   ' num2str((max(sum(~isnan(key_pressesf(:,:,110)),2)))) '   ' ...
       num2str((mean(sum(~isnan(key_pressesf(:,:,110)),2)))) '    ' num2str((std(sum(~isnan(key_pressesf(:,:,110)),2)))) '    ' num2str((median(sum(~isnan(key_pressesf(:,:,110)),2))))])
disp(['Survey time mean (minutes) (after filtering) = ' num2str(mean(X_filtered(:,16)/60,'omitnan'))]);
disp(['Survey time median (minutes) (after filtering) = ' num2str(median(X_filtered(:,16)/60,'omitnan'))]);
disp(['Survey time SD (minutes) (after filtering) = ' num2str(std(X_filtered(:,16)/60,'omitnan'))]);
disp(['First survey start date (before filtering) = ' datestr(min(X(:,17)))]);
disp(['Last survey end date (before filtering) = ' datestr(max(X(:,18)))]);
disp(['Gender, male respondents (after filtering)= ' num2str(sum(X_filtered(:,2)==2))])
disp(['Gender, female respondents (after filtering) = ' num2str(sum(X_filtered(:,2)==1))])
disp(['Gender, I prefer not to respond (after filtering) = ' num2str(sum(isnan(X_filtered(:,2))))])
disp(['Age (after filtering), Mean = ' num2str(mean(X_filtered(:,3),'omitnan'))])
disp(['Age (after filtering), SD = ' num2str(std(X_filtered(:,3),'omitnan'))])
%% Most common countries (after filtering)
[~, ~, ub] = unique(Country_filtered);
test2counts = histcounts(ub, 'BinMethod','integers');
[B,I] = maxk(test2counts,10);
country_unique = unique(Country_filtered);
disp('Most common countries (after filtering) = ')
disp(country_unique(I)')
disp(B)
%% Figure 1. Distance of the automated vehicle (AV) to the pedestrian in the non-yielding and yielding trials.
CarPosDat    = readmatrix("car_position_data.txt");
CarPosDat(:,2) = 110 - CarPosDat(:,2);

time_NY = CarPosDat(53:275,1)-CarPosDat(53,1)+1;
PosNY   = CarPosDat(53:275,2);
Loc50   = find(PosNY<50,1,'first')+2; 
Loc35   = find(PosNY<35,1,'first')+2; 

time_Y  = CarPosDat(410:end,1)-CarPosDat(410,1)+1;
PosY    = CarPosDat(410:end,2);

time_Y(end+1)=11;PosY(end+1)=PosY(end); % fix end points

figure;hold on;grid on;box on
plot(time_Y,PosY,'g-','Linewidth',5);
plot(time_NY,PosNY,'r:','Linewidth',5);
plot(time_NY(Loc50),PosNY(Loc50),'o','color',[.5 .5 .5],'markersize',22,'markerfacecolor',[.5 .5 .5]);
plot(time_NY(Loc35),PosNY(Loc35),'s','color',[.8 .8 .8],'markersize',26,'markerfacecolor',[.8 .8 .8]);
rectangle('position',[0 0 1 110],'facecolor',[127 127 127 20]/255,'Linewidth',.5);
rectangle('position',[11 0 10 110],'facecolor',[127 127 127 20]/255,'Linewidth',.5);
xlabel('Time (s)')
ylabel('Distance between vehicle and pedestrian (m)')
set(gca,'LooseInset',[0.01 0.01 0.01 0.01],'xlim',[0 12],'xtick',0:12,'ylim',[0 110])
legend({'Yielding','Non-yielding','eHMI activation 50 m','eHMI activation 35 m'},'Location', 'northeast');
h=findobj('FontName','Helvetica'); set(h,'FontSize',16,'Fontname','Arial')
if savepng==1;set(gcf,'WindowState', 'maximized');print(gcf,'Figure1.png','-dpng','-r300');end
%% Colour
% Figure 4. Percentage of participants pressing the response key as a function of elapsed video time, distinguishing between yielding behaviour and eHMI colour. 
% 1. eHMI colour (1 = red, 2 = green, 3 = cyan)	
% 2. eHMI position (1 = bumper, 2 = windshield, 3 = roof)
% 3. AV behaviour (1 = non-yielding, 2 = yielding)
% 4. eHMI activation (1 = no, 2 = yes)	
% 5. eHMI activation distance (0 = no activation, 1 = 35 m, 2 = 50 m)
% 6. eHMI message (0 = no activation, 1 = DON'T WALK, 2 = WALK, 3 = WON'T STOP, 4 = WILL STOP, 5 = bar of light)	
% 7. Distraction (0 = no distraction, 1 = distraction)
% 8. Congruence (0 = congruent, 1 = incongruent)

time = (0.05:0.1:size(key_pressesf,3)/10)'; % Time vector
v1=find(video_info(:,3)==2 & video_info(:,1)==1 & video_info(:,5)>0 & video_info(:,8)==0); % yielding, Red
v2=find(video_info(:,3)==2 & video_info(:,1)==2 & video_info(:,5)>0 & video_info(:,8)==0); % yielding, Green
v3=find(video_info(:,3)==2 & video_info(:,1)==3 & video_info(:,5)>0 & video_info(:,8)==0); % yielding, Cyan
v4=find(video_info(:,3)==1 & video_info(:,1)==1 & video_info(:,5)>0 & video_info(:,8)==0); % non-yielding, Red
v5=find(video_info(:,3)==1 & video_info(:,1)==2 & video_info(:,5)>0 & video_info(:,8)==0); % non-yielding, Green
v6=find(video_info(:,3)==1 & video_info(:,1)==3 & video_info(:,5)>0 & video_info(:,8)==0); % non-yielding, Cyan
figure;hold on;grid on;box on
title('Effect of colour')
calculate_means
plot(time,V1,'-','Linewidth',5,'color',[255 2 0]/255);
plot(time,V2,'-','Linewidth',5,'color',[0 255 0]/255);
plot(time,V3,'-','Linewidth',5,'color',[127 255 215]/255);
plot(time,V4,':','Linewidth',5,'color',[255 2 0]/255);
plot(time,V5,':','Linewidth',5,'color',[0 255 0]/255);
plot(time,V6,':','Linewidth',5,'color',[127 255 215]/255);
xlabel('Time (s)')
ylabel('Percentage of participants pressing response key')
set(gca,'LooseInset',[0.01 0.01 0.01 0.01],'xlim',[0 12],'xtick',1:12,'ylim',[0 60])
legend(['Yielding - Red (' num2str(length(v1)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v1p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v1p,'omitnan')) '%)'],...
    ['Yielding - Green (' num2str(length(v2)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v2p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v2p,'omitnan')) '%)'],...
    ['Yielding - Cyan (' num2str(length(v3)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v3p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v3p,'omitnan')) '%)'],...
    ['Non-yielding - Red (' num2str(length(v4)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v4p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v4p,'omitnan')) '%)'],...
    ['Non-yielding - Green (' num2str(length(v5)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v5p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v5p,'omitnan')) '%)'],...
    ['Non-yielding - Cyan (' num2str(length(v6)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v6p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v6p,'omitnan')) '%)'],...
    'autoupdate','off');
h=findobj('FontName','Helvetica'); set(h,'FontSize',16,'Fontname','Arial')
rectangle('position',[0 0 1 100],'facecolor',[127 127 127 20]/255,'Linewidth',.5);
rectangle('position',[11 0 10 100],'facecolor',[127 127 127 20]/255,'Linewidth',.5);
V=1:10:111;
[pyielding1,pyielding2,pyielding3,pnonyielding1,pnonyielding2,pnonyielding3]=deal(NaN(1,length(V)));
for i=1:length(V)
    [~,pyielding1(i)]=ttest(mean(mean(key_pressesf(:,v1,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v2,V(i):V(i)+9),3,'omitnan'),2,'omitnan')); % RG
    [~,pyielding2(i)]=ttest(mean(mean(key_pressesf(:,v1,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v3,V(i):V(i)+9),3,'omitnan'),2,'omitnan')); % RC
    [~,pyielding3(i)]=ttest(mean(mean(key_pressesf(:,v2,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v3,V(i):V(i)+9),3,'omitnan'),2,'omitnan')); % GC
    
    if pyielding1(i)<0.005 && pyielding2(i)<0.005 && pyielding3(i)<0.005
        text(i-0.5,4,'Y (RG, RC, GC)','horizontalalignment','center','color','b')
    elseif pyielding1(i)<0.005 && pyielding2(i)<0.005 && pyielding3(i)>=0.005
        text(i-0.5,4,'Y (RG, RC)','horizontalalignment','center','color','b')
    elseif pyielding1(i)<0.005 && pyielding2(i)>0.005 && pyielding3(i)<0.005
        text(i-0.5,4,'Y (RG, GC)','horizontalalignment','center','color','b')
    elseif pyielding1(i)<0.005 && pyielding2(i)>0.005 && pyielding3(i)>=0.005
        text(i-0.5,4,'Y (RG)','horizontalalignment','center','color','b')
    elseif pyielding1(i)>0.005 && pyielding2(i)<0.005 && pyielding3(i)<0.005
        text(i-0.5,4,'Y (RC, GC)','horizontalalignment','center','color','b')
    elseif pyielding1(i)>0.005 && pyielding2(i)<0.005 && pyielding3(i)>=0.005
        text(i-0.5,4,'Y (RC)','horizontalalignment','center','color','b')
    elseif pyielding1(i)>0.005 && pyielding2(i)>0.005 && pyielding3(i)<0.005
        text(i-0.5,4,'Y (GC)','horizontalalignment','center','color','b')
    elseif pyielding1(i)>0.005 && pyielding2(i)>0.005 && pyielding3(i)>=0.005
        text(i-0.5,4,'','horizontalalignment','center','color','b')
    end
    [~,pnonyielding1(i)]=ttest(mean(mean(key_pressesf(:,v4,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v5,V(i):V(i)+9),3,'omitnan'),2,'omitnan'));
    [~,pnonyielding2(i)]=ttest(mean(mean(key_pressesf(:,v4,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v6,V(i):V(i)+9),3,'omitnan'),2,'omitnan'));
    [~,pnonyielding3(i)]=ttest(mean(mean(key_pressesf(:,v5,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v6,V(i):V(i)+9),3,'omitnan'),2,'omitnan'));
    if pnonyielding1(i)<0.005 && pnonyielding2(i)<0.005 && pnonyielding3(i)<0.005
        text(i-0.5,2,'NY (RG,RC,GC)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)<0.005 && pnonyielding2(i)<0.005 && pnonyielding3(i)>=0.005
        text(i-0.5,2,'NY (RG,RC)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)<0.005 && pnonyielding2(i)>0.005 && pnonyielding3(i)<0.005
        text(i-0.5,2,'NY (RG,GC)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)<0.005 && pnonyielding2(i)>0.005 && pnonyielding3(i)>=0.005
        text(i-0.5,2,'NY (RG)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)>0.005 && pnonyielding2(i)<0.005 && pnonyielding3(i)<0.005
        text(i-0.5,2,'NY (RC,GC)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)>0.005 && pnonyielding2(i)<0.005 && pnonyielding3(i)>=0.005
        text(i-0.5,2,'NY (RC)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)>0.005 && pnonyielding2(i)>0.005 && pnonyielding3(i)<0.005
        text(i-0.5,2,'NY (GC)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)>0.005 && pnonyielding2(i)>0.005 && pnonyielding3(i)>=0.005
        text(i-0.5,2,'','horizontalalignment','center','color','b')
    end
end
h=findobj('FontName','Helvetica'); set(h,'FontSize',16,'Fontname','Arial')
if savepng==1;set(gcf,'WindowState', 'maximized');print(gcf,'Figure4.png','-dpng','-r300');end
%% Message
% Figure 5. Percentage of participants pressing the response key as a function of elapsed video time, distinguishing between yielding behaviour and eHMI message.
% 1. eHMI colour (1 = red, 2 = green, 3 = cyan)	
% 2. eHMI position (1 = bumper, 2 = windshield, 3 = roof)
% 3. AV behaviour (1 = non-yielding, 2 = yielding)
% 4. eHMI activation (1 = no, 2 = yes)	
% 5. eHMI activation distance (0 = no activation, 1 = 35 m, 2 = 50 m)
% 6. eHMI message (0 = no activation, 1 = DON'T WALK, 2 = WALK, 3 = WON'T STOP, 4 = WILL STOP, 5 = bar of light)	
% 7. Distraction (0 = no distraction, 1 = distraction)
% 8. Congruence (0 = congruent, 1 = incongruent)
v1=find(video_info(:,3)==2 & video_info(:,6)==2 & video_info(:,5)>0 & video_info(:,8)==0); % Yielding, Walk
v2=find(video_info(:,3)==2 & video_info(:,6)==4 & video_info(:,5)>0 & video_info(:,8)==0); % Yielding, Will stop
v3=find(video_info(:,3)==2 & video_info(:,6)==5 & video_info(:,5)>0 & video_info(:,8)==0); % Yielding, Bar
v4=find(video_info(:,3)==1 & video_info(:,6)==1 & video_info(:,5)>0 & video_info(:,8)==0); % Non-yielding, Don't walk
v5=find(video_info(:,3)==1 & video_info(:,6)==3 & video_info(:,5)>0 & video_info(:,8)==0); % Non-yielding, Won't stop
v6=find(video_info(:,3)==1 & video_info(:,6)==5 & video_info(:,5)>0 & video_info(:,8)==0); % Non-yielding, Bar
figure;hold on;grid on;box on
title('Effect of message')
calculate_means
plot(time,V1,'k','Linewidth',5);
plot(time,V2,'m','Linewidth',5);
plot(time,V3,'c','Linewidth',5);
plot(time,V4,'k:','Linewidth',5);
plot(time,V5,'m:','Linewidth',5);
plot(time,V6,'c:','Linewidth',5);
xlabel('Time (s)')
ylabel('Percentage of participants pressing response key')
set(gca,'LooseInset',[0.01 0.01 0.01 0.01],'xlim',[0 12],'xtick',1:12,'ylim',[0 60])
legend(['Yielding - Egocentric (WALK) (' num2str(length(v1)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v1p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v1p,'omitnan')) '%)'],...
    ['Yielding - Allocentric (WILL STOP) (' num2str(length(v2)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v2p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v2p,'omitnan')) '%)'],...
    ['Yielding - Bar (' num2str(length(v3)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v3p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v3p,'omitnan')) '%)'],...
    ['Non-yielding - Egocentric (DON''T WALK) (' num2str(length(v4)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v4p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v4p,'omitnan')) '%)'],...
    ['Non-yielding - Allocentric (WON''T STOP) (' num2str(length(v5)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v5p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v5p,'omitnan')) '%)'],...
    ['Non-yielding - Bar (' num2str(length(v6)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v6p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v6p,'omitnan')) '%)'],...
    'autoupdate','off');
h=findobj('FontName','Helvetica'); set(h,'FontSize',16,'Fontname','Arial')
rectangle('position',[0 0 1 100],'facecolor',[127 127 127 20]/255,'Linewidth',.5);
rectangle('position',[11 0 10 100],'facecolor',[127 127 127 20]/255,'Linewidth',.5);
[pyielding1,pyielding2,pyielding3,pnonyielding1,pnonyielding2,pnonyielding3]=deal(NaN(1,length(V)));
for i=1:length(V)
    [~,pyielding1(i)]=ttest(mean(mean(key_pressesf(:,v1,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v2,V(i):V(i)+9),3,'omitnan'),2,'omitnan')); % EA
    [~,pyielding2(i)]=ttest(mean(mean(key_pressesf(:,v1,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v3,V(i):V(i)+9),3,'omitnan'),2,'omitnan')); % EB
    [~,pyielding3(i)]=ttest(mean(mean(key_pressesf(:,v2,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v3,V(i):V(i)+9),3,'omitnan'),2,'omitnan')); % AB
    
    if pyielding1(i)<0.005 && pyielding2(i)<0.005 && pyielding3(i)<0.005
        text(i-0.5,4,'Y (EA,EB,AB)','horizontalalignment','center','color','b')
    elseif pyielding1(i)<0.005 && pyielding2(i)<0.005 && pyielding3(i)>=0.005
        text(i-0.5,4,'Y (EA,EB)','horizontalalignment','center','color','b')
    elseif pyielding1(i)<0.005 && pyielding2(i)>0.005 && pyielding3(i)<0.005
        text(i-0.5,4,'Y (EA,AB)','horizontalalignment','center','color','b')
    elseif pyielding1(i)<0.005 && pyielding2(i)>0.005 && pyielding3(i)>=0.005
        text(i-0.5,4,'Y (EA)','horizontalalignment','center','color','b')
    elseif pyielding1(i)>0.005 && pyielding2(i)<0.005 && pyielding3(i)<0.005
        text(i-0.5,4,'Y (EB,AB)','horizontalalignment','center','color','b')
    elseif pyielding1(i)>0.005 && pyielding2(i)<0.005 && pyielding3(i)>=0.005
        text(i-0.5,4,'Y (EB)','horizontalalignment','center','color','b')
    elseif pyielding1(i)>0.005 && pyielding2(i)>0.005 && pyielding3(i)<0.005
        text(i-0.5,4,'Y (AB)','horizontalalignment','center','color','b')
    elseif pyielding1(i)>0.005 && pyielding2(i)>0.005 && pyielding3(i)>=0.005
        text(i-0.5,4,'','horizontalalignment','center','color','b')
    end
    [~,pnonyielding1(i)]=ttest(mean(mean(key_pressesf(:,v4,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v5,V(i):V(i)+9),3,'omitnan'),2,'omitnan')); % EA
    [~,pnonyielding2(i)]=ttest(mean(mean(key_pressesf(:,v4,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v6,V(i):V(i)+9),3,'omitnan'),2,'omitnan')); % EB
    [~,pnonyielding3(i)]=ttest(mean(mean(key_pressesf(:,v5,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v6,V(i):V(i)+9),3,'omitnan'),2,'omitnan')); % AB
    if pnonyielding1(i)<0.005 && pnonyielding2(i)<0.005 && pnonyielding3(i)<0.005
        text(i-0.5,2,'NY (EA,EB,AB)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)<0.005 && pnonyielding2(i)<0.005 && pnonyielding3(i)>=0.005
        text(i-0.5,2,'NY (EA,EB)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)<0.005 && pnonyielding2(i)>0.005 && pnonyielding3(i)<0.005
        text(i-0.5,2,'NY (EA,AB)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)<0.005 && pnonyielding2(i)>0.005 && pnonyielding3(i)>=0.005
        text(i-0.5,2,'NY (EA)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)>0.005 && pnonyielding2(i)<0.005 && pnonyielding3(i)<0.005
        text(i-0.5,2,'NY (EB,AB)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)>0.005 && pnonyielding2(i)<0.005 && pnonyielding3(i)>=0.005
        text(i-0.5,2,'NY (EB)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)>0.005 && pnonyielding2(i)>0.005 && pnonyielding3(i)<0.005
        text(i-0.5,2,'NY (AB)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)>0.005 && pnonyielding2(i)>0.005 && pnonyielding3(i)>=0.005
        text(i-0.5,2,'','horizontalalignment','center','color','b')
    end
end
h=findobj('FontName','Helvetica'); set(h,'FontSize',16,'Fontname','Arial')
if savepng==1;set(gcf,'WindowState', 'maximized');print(gcf,'Figure5.png','-dpng','-r300');end
%% Message with 'best' colours
% Figure 6. Percentage of participants pressing the response key as a function of elapsed video time, distinguishing between eHMI message, selecting green and cyan eHMIs for yielding vehicles and red eHMIs for non-yielding vehicles. 

% 1. eHMI colour (1 = red, 2 = green, 3 = cyan)	
% 2. eHMI position (1 = bumper, 2 = windshield, 3 = roof)
% 3. AV behaviour (1 = non-yielding, 2 = yielding)
% 4. eHMI activation (1 = no, 2 = yes)	
% 5. eHMI activation distance (0 = no activation, 1 = 35 m, 2 = 50 m)
% 6. eHMI message (0 = no activation, 1 = DON'T WALK, 2 = WALK, 3 = WON'T STOP, 4 = WILL STOP, 5 = bar of light)	
% 7. Distraction (0 = no distraction, 1 = distraction)
% 8. Congruence (0 = congruent, 1 = incongruent)

vby=find(video_info(:,3)==2 & video_info(:,5)==0 & video_info(:,8)==0); % yielding, eHMI inactive
vbny=find(video_info(:,3)==1 & video_info(:,5)==0 & video_info(:,8)==0); % non-yielding, eHMI inactive
VBY = 100 * squeeze(mean(mean(key_pressesf(:,vby,:),2,'omitnan'),1,'omitnan'));
VBNY = 100 * squeeze(mean(mean(key_pressesf(:,vbny,:),2,'omitnan'),1,'omitnan'));

v1=find(video_info(:,3)==1 & video_info(:,1)==1 & video_info(:,5)>0 & video_info(:,8)==0 & video_info(:,6)==1); % Non-yielding, Red
v2=find(video_info(:,3)==1 & video_info(:,1)==1 & video_info(:,5)>0 & video_info(:,8)==0 & video_info(:,6)==3); % Non-yielding, Red
v3=find(video_info(:,3)==1 & video_info(:,1)==1 & video_info(:,5)>0 & video_info(:,8)==0 & video_info(:,6)==5); % Non-yielding, Red
v4=find(video_info(:,3)==2 & video_info(:,1)>1 & video_info(:,5)>0 & video_info(:,8)==0 & video_info(:,6)==2); % Yielding, Green or Cyan
v5=find(video_info(:,3)==2 & video_info(:,1)>1 & video_info(:,5)>0 & video_info(:,8)==0 & video_info(:,6)==4); % Yielding, Green or Cyan
v6=find(video_info(:,3)==2 & video_info(:,1)>1 & video_info(:,5)>0 & video_info(:,8)==0 & video_info(:,6)==5); % Yielding, Green or Cyan

figure;hold on;grid on;box on
title('Effect of message (for the ''best'' colours)')
calculate_means
vbyp=100 * mean(mean(key_pressesf(:,vby,51:110),3,'omitnan'),2,'omitnan');
vbnyp=100 * mean(mean(key_pressesf(:,vbny,51:110),3,'omitnan'),2,'omitnan');

plot(time,V4,'-','Linewidth',5,'color',[0 255 0]/255);
plot(time,V5,':','Linewidth',5,'color',[0 255 0]/255);
plot(time,V6,'-','Linewidth',2,'color',[0 255 0]/255);
plot(time,VBY,'-','Linewidth',5,'color',[0.5 0.5 0.5]);
plot(time,V1,'-','Linewidth',5,'color',[255 2 0]/255);
plot(time,V2,':','Linewidth',5,'color',[255 2 0]/255);
plot(time,V3,'-','Linewidth',2,'color',[255 2 0]/255);
plot(time,VBNY,':','Linewidth',5,'color',[0.5 0.5 0.5]);

xlabel('Time (s)')
ylabel('Percentage of participants pressing response key')
set(gca,'LooseInset',[0.01 0.01 0.01 0.01],'xlim',[0 12],'xtick',1:12,'ylim',[0 60])
legend(['Yielding - Green & cyan - Egocentric (WALK) (' num2str(length(v4)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v4p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v4p,'omitnan')) '%)'],...
    ['Yielding - Green & cyan - Allocentric (WILL STOP) (' num2str(length(v5)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v5p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v5p,'omitnan')) '%)'],...
    ['Yielding - Green & cyan - Bar (' num2str(length(v6)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v6p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v6p,'omitnan')) '%)'],...
    ['Yielding - Inactive (' num2str(length(vby)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(vbyp,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(vbyp,'omitnan')) '%)'],...
    ['Non-yielding - Red - Egocentric (DON''T WALK) (' num2str(length(v1)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v1p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v1p,'omitnan')) '%)'],...
    ['Non-yielding - Red - Allocentric (WON''T STOP) (' num2str(length(v2)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v2p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v2p,'omitnan')) '%)'],...
    ['Non-yielding - Red - Bar (' num2str(length(v3)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v3p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v3p,'omitnan')) '%)'],...
    ['Non-yielding - Inactive (' num2str(length(vbny)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(vbnyp,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(vbnyp,'omitnan')) '%)'],...
    'autoupdate','off');

h=findobj('FontName','Helvetica'); set(h,'FontSize',16,'Fontname','Arial')
rectangle('position',[0 0 1 100],'facecolor',[127 127 127 20]/255,'Linewidth',.5);
rectangle('position',[11 0 10 100],'facecolor',[127 127 127 20]/255,'Linewidth',.5);
[pyielding1,pyielding2,pyielding3,pnonyielding1,pnonyielding2,pnonyielding3]=deal(NaN(1,length(V)));
for i=1:length(V)
    [~,pyielding1(i)]=ttest(mean(mean(key_pressesf(:,v4,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v5,V(i):V(i)+9),3,'omitnan'),2,'omitnan')); % EA
    [~,pyielding2(i)]=ttest(mean(mean(key_pressesf(:,v4,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v6,V(i):V(i)+9),3,'omitnan'),2,'omitnan')); % EB
    [~,pyielding3(i)]=ttest(mean(mean(key_pressesf(:,v5,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v6,V(i):V(i)+9),3,'omitnan'),2,'omitnan')); % AB
    if pyielding1(i)<0.005 && pyielding2(i)<0.005 && pyielding3(i)<0.005
        text(i-0.5,4,'Y (EA,EB,AB)','horizontalalignment','center','color','b')
    elseif pyielding1(i)<0.005 && pyielding2(i)<0.005 && pyielding3(i)>=0.005 
        text(i-0.5,4,'Y (EA,EB)','horizontalalignment','center','color','b')
    elseif pyielding1(i)<0.005 && pyielding2(i)>0.005 && pyielding3(i)<0.005
        text(i-0.5,4,'Y (EA,AB)','horizontalalignment','center','color','b')
    elseif pyielding1(i)<0.005 && pyielding2(i)>0.005 && pyielding3(i)>=0.005
        text(i-0.5,4,'Y (EA)','horizontalalignment','center','color','b')
    elseif pyielding1(i)>0.005 && pyielding2(i)<0.005 && pyielding3(i)<0.005
        text(i-0.5,4,'Y (EB,AB)','horizontalalignment','center','color','b')
    elseif pyielding1(i)>0.005 && pyielding2(i)<0.005 && pyielding3(i)>=0.005
        text(i-0.5,4,'Y (EB)','horizontalalignment','center','color','b')
    elseif pyielding1(i)>0.005 && pyielding2(i)>0.005 && pyielding3(i)<0.005
        text(i-0.5,4,'Y (AB)','horizontalalignment','center','color','b')
    elseif pyielding1(i)>0.005 && pyielding2(i)>0.005 && pyielding3(i)>=0.005
        text(i-0.5,4,'','horizontalalignment','center','color','b')
    end
    [~,pnonyielding1(i)]=ttest(mean(mean(key_pressesf(:,v1,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v2,V(i):V(i)+9),3,'omitnan'),2,'omitnan')); % EA
    [~,pnonyielding2(i)]=ttest(mean(mean(key_pressesf(:,v1,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v3,V(i):V(i)+9),3,'omitnan'),2,'omitnan')); % EB
    [~,pnonyielding3(i)]=ttest(mean(mean(key_pressesf(:,v2,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v3,V(i):V(i)+9),3,'omitnan'),2,'omitnan')); % AB
    if pnonyielding1(i)<0.005 && pnonyielding2(i)<0.005 && pnonyielding3(i)<0.005
        text(i-0.5,2,'NY (EA,EB,AB)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)<0.005 && pnonyielding2(i)<0.005 && pnonyielding3(i)>=0.005
        text(i-0.5,2,'NY (EA,EB)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)<0.005 && pnonyielding2(i)>0.005 && pnonyielding3(i)<0.005
        text(i-0.5,2,'NY (EA,AB)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)<0.005 && pnonyielding2(i)>0.005 && pnonyielding3(i)>=0.005
        text(i-0.5,2,'NY (EA)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)>0.005 && pnonyielding2(i)<0.005 && pnonyielding3(i)<0.005
        text(i-0.5,2,'NY (EB,AB)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)>0.005 && pnonyielding2(i)<0.005 && pnonyielding3(i)>=0.005
        text(i-0.5,2,'NY (EB)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)>0.005 && pnonyielding2(i)>0.005 && pnonyielding3(i)<0.005
        text(i-0.5,2,'NY (AB)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)>0.005 && pnonyielding2(i)>0.005 && pnonyielding3(i)>=0.005
        text(i-0.5,2,'','horizontalalignment','center','color','b')
    end
end
h=findobj('FontName','Helvetica'); set(h,'FontSize',16,'Fontname','Arial')
if savepng==1;set(gcf,'WindowState', 'maximized');print(gcf,'Figure6.png','-dpng','-r300');end
%% Country analysis
% Figure 11. Scatter plots of mean perceived-safety percentages for participants from different countries, for the conditions shown in Fig.=yre 6.
CA=[v4p v5p v6p vbyp v1p v2p v3p vbnyp]; % Use the data from Figure 6 for the country analysis
CAM=[mean(CA(strcmp(Country_filtered,'VEN'),:),'omitnan')' mean(CA(strcmp(Country_filtered,'USA'),:),'omitnan')' mean(CA(strcmp(Country_filtered,'IND'),:),'omitnan')' ...
     mean(CA(strcmp(Country_filtered,'TUR'),:),'omitnan')']; % means per country

figure;hold on
subplot(1,3,1); % USA vs. VEN
plot(CAM(1,1),CAM(1,2),'s','linewidth',3,'markeredgecolor',[0 255 0]/255,'markerfacecolor',[0 255 0]/255,'markersize',11);hold on
plot(CAM(2,1),CAM(2,2),'o','linewidth',3,'markeredgecolor',[0 255 0]/255,'markerfacecolor',[0 255 0]/255,'markersize',11);hold on
plot(CAM(3,1),CAM(3,2),'d','linewidth',3,'markeredgecolor',[0 255 0]/255,'markerfacecolor',[0 255 0]/255,'markersize',11);hold on
plot(CAM(4,1),CAM(4,2),'^','linewidth',3,'markeredgecolor',[0.5 0.5 0.5],'markerfacecolor',[0.5 0.5 0.5],'markersize',11);hold on
plot(CAM(5,1),CAM(5,2),'s','linewidth',3,'markeredgecolor',[255 2 0]/255,'markerfacecolor','none','markersize',11);hold on 
plot(CAM(6,1),CAM(6,2),'o','linewidth',3,'markeredgecolor',[255 2 0]/255,'markerfacecolor','none','markersize',11);hold on
plot(CAM(7,1),CAM(7,2),'d','linewidth',3,'markeredgecolor',[255 2 0]/255,'markerfacecolor','none','markersize',11);hold on
plot(CAM(8,1),CAM(8,2),'^','linewidth',3,'markeredgecolor',[0.5 0.5 0.5],'markerfacecolor','none','markersize',11);hold on
text(20,42,['\itr\rm = ' sprintf('%0.2f',corr(CAM(:,1),CAM(:,2)))])
plot([0 100],[0 100],'k--')
xlabel('Perceived-safety percentage - Respondents from Venezuela')
ylabel('Perceived-safety percentage - Respondents from USA')
axis equal
set(gca,'xlim',[19 43],'ylim',[19 43],'TickDir','out','ticklength',[0.005 0.005],'LooseInset',[0.01 0.01 0.01 0.01])

subplot(1,3,2) % IND vs. VEN
plot(CAM(1,1),CAM(1,3),'s','linewidth',3,'markeredgecolor',[0 255 0]/255,'markerfacecolor',[0 255 0]/255,'markersize',11);hold on
plot(CAM(2,1),CAM(2,3),'o','linewidth',3,'markeredgecolor',[0 255 0]/255,'markerfacecolor',[0 255 0]/255,'markersize',11);hold on
plot(CAM(3,1),CAM(3,3),'d','linewidth',3,'markeredgecolor',[0 255 0]/255,'markerfacecolor',[0 255 0]/255,'markersize',11);hold on
plot(CAM(4,1),CAM(4,3),'^','linewidth',3,'markeredgecolor',[0.5 0.5 0.5],'markerfacecolor',[0.5 0.5 0.5],'markersize',11);hold on
plot(CAM(5,1),CAM(5,3),'s','linewidth',3,'markeredgecolor',[255 2 0]/255,'markerfacecolor','none','markersize',11);hold on 
plot(CAM(6,1),CAM(6,3),'o','linewidth',3,'markeredgecolor',[255 2 0]/255,'markerfacecolor','none','markersize',11);hold on
plot(CAM(7,1),CAM(7,3),'d','linewidth',3,'markeredgecolor',[255 2 0]/255,'markerfacecolor','none','markersize',11);hold on
plot(CAM(8,1),CAM(8,3),'^','linewidth',3,'markeredgecolor',[0.5 0.5 0.5],'markerfacecolor','none','markersize',11);hold on
text(20,42,['\itr\rm = ' sprintf('%0.2f',corr(CAM(:,1),CAM(:,3)))])
plot([0 100],[0 100],'k--')
xlabel('Perceived-safety percentage - Respondents from Venezuela')
ylabel('Perceived-safety percentage - Respondents from India')
axis equal
set(gca,'xlim',[19 43],'ylim',[19 43],'TickDir','out','ticklength',[0.005 0.005],'LooseInset',[0.01 0.01 0.01 0.01])

subplot(1,3,3) % TUR vs. VEN
plot(CAM(1,1),CAM(1,4),'s','linewidth',3,'markeredgecolor',[0 255 0]/255,'markerfacecolor',[0 255 0]/255,'markersize',11);hold on
plot(CAM(2,1),CAM(2,4),'o','linewidth',3,'markeredgecolor',[0 255 0]/255,'markerfacecolor',[0 255 0]/255,'markersize',11);hold on
plot(CAM(3,1),CAM(3,4),'d','linewidth',3,'markeredgecolor',[0 255 0]/255,'markerfacecolor',[0 255 0]/255,'markersize',11);hold on
plot(CAM(4,1),CAM(4,4),'^','linewidth',3,'markeredgecolor',[0.5 0.5 0.5],'markerfacecolor',[0.5 0.5 0.5],'markersize',11);hold on
plot(CAM(5,1),CAM(5,4),'s','linewidth',3,'markeredgecolor',[255 2 0]/255,'markerfacecolor','none','markersize',11);hold on 
plot(CAM(6,1),CAM(6,4),'o','linewidth',3,'markeredgecolor',[255 2 0]/255,'markerfacecolor','none','markersize',11);hold on
plot(CAM(7,1),CAM(7,4),'d','linewidth',3,'markeredgecolor',[255 2 0]/255,'markerfacecolor','none','markersize',11);hold on
plot(CAM(8,1),CAM(8,4),'^','linewidth',3,'markeredgecolor',[0.5 0.5 0.5],'markerfacecolor','none','markersize',11);hold on
text(20,42,['\itr\rm = ' sprintf('%0.2f',corr(CAM(:,1),CAM(:,4)))])
plot([0 100],[0 100],'k--')
xlabel('Perceived-safety percentage - Respondents from Venezuela')
ylabel('Perceived-safety percentage - Respondents from Turkey')
axis equal
set(gca,'xlim',[19 43],'ylim',[19 43],'TickDir','out','ticklength',[0.005 0.005],'LooseInset',[0.01 0.01 0.01 0.01])
h=findobj('FontName','Helvetica'); set(h,'FontSize',11,'Fontname','Arial')
h=legend('Yielding - Green & cyan - Egocentric (WALK)',...
    'Yielding - Green & cyan - Allocentric (WILL STOP)',...
    'Yielding - Green & cyan - Bar',...
    'Yielding - Inactive',...
    'Non-yielding - Red - Egocentric (DON''T WALK)',...
    'Non-yielding - Red - Allocentric (WON''T STOP)',...
    'Non-yielding - Red - Bar',...
    'Non-yielding - Inactive',...
    'autoupdate','off');
h.NumColumns = 2;
set(h,'Position',[0.42 0.66 0.21 0.3],'fontsize',11)
legend boxoff
if savepng==1;set(gcf,'WindowState', 'maximized');print(gcf,'Figure11.png','-dpng','-r300');end
%% Position
% Figure 7. Percentage of participants pressing the response key as a function of elapsed video time, distinguishing between yielding behaviour and eHMI position.

% 1. eHMI colour (1 = red, 2 = green, 3 = cyan)	
% 2. eHMI position (1 = bumper, 2 = windshield, 3 = roof)
% 3. AV behaviour (1 = non-yielding, 2 = yielding)
% 4. eHMI activation (1 = no, 2 = yes)	
% 5. eHMI activation distance (0 = no activation, 1 = 35 m, 2 = 50 m)
% 6. eHMI message (0 = no activation, 1 = DON'T WALK, 2 = WALK, 3 = WON'T STOP, 4 = WILL STOP, 5 = bar of light)	
% 7. Distraction (0 = no distraction, 1 = distraction)
% 8. Congruence (0 = congruent, 1 = incongruent)

v1=find(video_info(:,3)==2 & video_info(:,2)==1 & video_info(:,5)>0 & video_info(:,8)==0); % Yielding, Bumper
v2=find(video_info(:,3)==2 & video_info(:,2)==2 & video_info(:,5)>0 & video_info(:,8)==0); % Yielding, Windshield
v3=find(video_info(:,3)==2 & video_info(:,2)==3 & video_info(:,5)>0 & video_info(:,8)==0); % Yielding, Roof
v4=find(video_info(:,3)==1 & video_info(:,2)==1 & video_info(:,5)>0 & video_info(:,8)==0); % Non-yielding, Bumper
v5=find(video_info(:,3)==1 & video_info(:,2)==2 & video_info(:,5)>0 & video_info(:,8)==0); % Non-yielding, Windshield
v6=find(video_info(:,3)==1 & video_info(:,2)==3 & video_info(:,5)>0 & video_info(:,8)==0); % Non-yielding, Roof
figure;hold on;grid on;box on
title('Effect of position')
calculate_means
plot(time,V1,'k','Linewidth',5);
plot(time,V2,'m','Linewidth',5);
plot(time,V3,'c','Linewidth',5);
plot(time,V4,'k:','Linewidth',5);
plot(time,V5,'m:','Linewidth',5);
plot(time,V6,'c:','Linewidth',5);
xlabel('Time (s)')
ylabel('Percentage of participants pressing response key')
set(gca,'LooseInset',[0.01 0.01 0.01 0.01],'xlim',[0 12],'xtick',1:12,'ylim',[0 60])
legend(['Yielding - Bumper (' num2str(length(v1)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v1p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v1p,'omitnan')) '%)'],...
    ['Yielding - Windshield (' num2str(length(v2)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v2p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v2p,'omitnan')) '%)'],...
    ['Yielding - Roof (' num2str(length(v3)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v3p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v3p,'omitnan')) '%)'],...
    ['Non-yielding - Bumper (' num2str(length(v4)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v4p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v4p,'omitnan')) '%)'],...
    ['Non-yielding - Windshield (' num2str(length(v5)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v5p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v5p,'omitnan')) '%)'],...
    ['Non-yielding - Roof (' num2str(length(v6)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v6p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v6p,'omitnan')) '%)'],...
    'autoupdate','off');
h=findobj('FontName','Helvetica'); set(h,'FontSize',16,'Fontname','Arial')
rectangle('position',[0 0 1 100],'facecolor',[127 127 127 20]/255,'Linewidth',.5);
rectangle('position',[11 0 10 100],'facecolor',[127 127 127 20]/255,'Linewidth',.5);
[pyielding1,pyielding2,pyielding3,pnonyielding1,pnonyielding2,pnonyielding3]=deal(NaN(1,length(V)));
for i=1:length(V)
    [~,pyielding1(i)]=ttest(mean(mean(key_pressesf(:,v1,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v2,V(i):V(i)+9),3,'omitnan'),2,'omitnan'));
    [~,pyielding2(i)]=ttest(mean(mean(key_pressesf(:,v1,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v3,V(i):V(i)+9),3,'omitnan'),2,'omitnan'));
    [~,pyielding3(i)]=ttest(mean(mean(key_pressesf(:,v2,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v3,V(i):V(i)+9),3,'omitnan'),2,'omitnan'));
    if pyielding1(i)<0.005 && pyielding2(i)<0.005 && pyielding3(i)<0.005
        text(i-0.5,4,'Y (BW,BR,WR)','horizontalalignment','center','color','b')
    elseif pyielding1(i)<0.005 && pyielding2(i)<0.005 && pyielding3(i)>=0.005
        text(i-0.5,4,'Y (BW,BR)','horizontalalignment','center','color','b')
    elseif pyielding1(i)<0.005 && pyielding2(i)>0.005 && pyielding3(i)<0.005
        text(i-0.5,4,'Y (BW,WR)','horizontalalignment','center','color','b')
    elseif pyielding1(i)<0.005 && pyielding2(i)>0.005 && pyielding3(i)>=0.005
        text(i-0.5,4,'Y (BW)','horizontalalignment','center','color','b')
    elseif pyielding1(i)>0.005 && pyielding2(i)<0.005 && pyielding3(i)<0.005
        text(i-0.5,4,'Y (BR,WR)','horizontalalignment','center','color','b')
    elseif pyielding1(i)>0.005 && pyielding2(i)<0.005 && pyielding3(i)>=0.005
        text(i-0.5,4,'Y (BR)','horizontalalignment','center','color','b')
    elseif pyielding1(i)>0.005 && pyielding2(i)>0.005 && pyielding3(i)<0.005
        text(i-0.5,4,'Y (WR)','horizontalalignment','center','color','b')
    elseif pyielding1(i)>0.005 && pyielding2(i)>0.005 && pyielding3(i)>=0.005
        text(i-0.5,4,'','horizontalalignment','center','color','b')
    end
    [~,pnonyielding1(i)]=ttest(mean(mean(key_pressesf(:,v4,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v5,V(i):V(i)+9),3,'omitnan'),2,'omitnan'));
    [~,pnonyielding2(i)]=ttest(mean(mean(key_pressesf(:,v4,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v6,V(i):V(i)+9),3,'omitnan'),2,'omitnan'));
    [~,pnonyielding3(i)]=ttest(mean(mean(key_pressesf(:,v5,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v6,V(i):V(i)+9),3,'omitnan'),2,'omitnan'));
    if pnonyielding1(i)<0.005 && pnonyielding2(i)<0.005 && pnonyielding3(i)<0.005
        text(i-0.5,2,'NY (BW,BR,WR)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)<0.005 && pnonyielding2(i)<0.005 && pnonyielding3(i)>=0.005
        text(i-0.5,2,'NY (BW,BR)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)<0.005 && pnonyielding2(i)>0.005 && pnonyielding3(i)<0.005
        text(i-0.5,2,'NY (BW,WR)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)<0.005 && pnonyielding2(i)>0.005 && pnonyielding3(i)>=0.005
        text(i-0.5,2,'NY (BW)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)>0.005 && pnonyielding2(i)<0.005 && pnonyielding3(i)<0.005
        text(i-0.5,2,'NY (BR,WR)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)>0.005 && pnonyielding2(i)<0.005 && pnonyielding3(i)>=0.005
        text(i-0.5,2,'NY (BR)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)>0.005 && pnonyielding2(i)>0.005 && pnonyielding3(i)<0.005
        text(i-0.5,2,'NY (WR)','horizontalalignment','center','color','b')
    elseif pnonyielding1(i)>0.005 && pnonyielding2(i)>0.005 && pnonyielding3(i)>=0.005
        text(i-0.5,2,'','horizontalalignment','center','color','b')
    end
end
h=findobj('FontName','Helvetica'); set(h,'FontSize',16,'Fontname','Arial')
if savepng==1;set(gcf,'WindowState', 'maximized');print(gcf,'Figure7.png','-dpng','-r300');end
%% Activation distance, 35 and 50 m
% Figure 8. Percentage of participants pressing the response key as a function of elapsed video time, distinguishing between yielding behaviour and eHMI activation distance. 

% 1. eHMI colour (1 = red, 2 = green, 3 = cyan)	
% 2. eHMI position (1 = bumper, 2 = windshield, 3 = roof)
% 3. AV behaviour (1 = non-yielding, 2 = yielding)
% 4. eHMI activation (1 = no, 2 = yes)	
% 5. eHMI activation distance (0 = no activation, 1 = 35 m, 2 = 50 m)
% 6. eHMI message (0 = no activation, 1 = DON'T WALK, 2 = WALK, 3 = WON'T STOP, 4 = WILL STOP, 5 = bar of light)	
% 7. Distraction (0 = no distraction, 1 = distraction)
% 8. Congruence (0 = congruent, 1 = incongruent)

clear v5 v6
v1=find(video_info(:,3)==2 & video_info(:,5)==1 & video_info(:,8)==0); % Yielding, 35 m
v2=find(video_info(:,3)==2 & video_info(:,5)==2 & video_info(:,8)==0); % Yielding, 50 m
v3=find(video_info(:,3)==1 & video_info(:,5)==1 & video_info(:,8)==0); % Non-yielding, 35 m
v4=find(video_info(:,3)==1 & video_info(:,5)==2 & video_info(:,8)==0); % Non-yielding, 50 m
figure;hold on;grid on;box on
title('Effect of activation distance')
calculate_means
plot(time,V1,'-','Linewidth',5,'color',[0 0 0]);
plot(time,V2,'-','Linewidth',5,'color',[1 0 1]);
plot(time,V3,':','Linewidth',5,'color',[0 0 0]);
plot(time,V4,':','Linewidth',5,'color',[1 0 1]);

xlabel('Time (s)')
ylabel('Percentage of participants pressing response key')
set(gca,'LooseInset',[0.01 0.01 0.01 0.01],'xlim',[0 12],'xtick',1:12,'ylim',[0 60])
legend(['Yielding - 35 m (' num2str(length(v1)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v1p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v1p,'omitnan')) '%)'],...
    ['Yielding - 50 m (' num2str(length(v2)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v2p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v2p,'omitnan')) '%)'],...
    ['Non-yielding - 35 m (' num2str(length(v3)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v3p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v3p,'omitnan')) '%)'],...
    ['Non-yielding - 50 m (' num2str(length(v4)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v4p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v4p,'omitnan')) '%)'],'autoupdate','off');
rectangle('position',[0 0 1 100],'facecolor',[127 127 127 20]/255,'Linewidth',.5);
rectangle('position',[11 0 10 100],'facecolor',[127 127 127 20]/255,'Linewidth',.5);
[pyielding,pnonyielding]=deal(NaN(1,length(V)));
for i=1:length(V)
    [~,pyielding(i)]=ttest(mean(mean(key_pressesf(:,v1,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v2,V(i):V(i)+9),3,'omitnan'),2,'omitnan')); % Y (significance of difference between activation distances was observed for yielding vehicles)
    if pyielding(i)<0.005
        text(i-0.5,4,'Y','horizontalalignment','center','color','b')
    end
    [~,pnonyielding(i)]=ttest(mean(mean(key_pressesf(:,v3,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v4,V(i):V(i)+9),3,'omitnan'),2,'omitnan')); % NY (significance of difference between activation distances was observed for non-yielding vehicles)
    if pnonyielding(i)<0.005
        text(i-0.5,2,'NY','horizontalalignment','center','color','b')
    end
end
h=findobj('FontName','Helvetica'); set(h,'FontSize',16,'Fontname','Arial')
if savepng==1;set(gcf,'WindowState', 'maximized');print(gcf,'Figure8.png','-dpng','-r300');end
%% eHMI activation distance for different messages (yielding AVs only)
% Figure 9. Percentage of participants pressing the response key as a function of elapsed video time in videos depicting a yielding AV, comparing early and late eHMI activation distance, for red bar messages and green/cyan WALK messages.

% 1. eHMI colour (1 = red, 2 = green, 3 = cyan)	
% 2. eHMI position (1 = bumper, 2 = windshield, 3 = roof)
% 3. AV behaviour (1 = non-yielding, 2 = yielding)
% 4. eHMI activation (1 = no, 2 = yes)	
% 5. eHMI activation distance (0 = no activation, 1 = 35 m, 2 = 50 m)
% 6. eHMI message (0 = no activation, 1 = DON'T WALK, 2 = WALK, 3 = WON'T STOP, 4 = WILL STOP, 5 = bar of light)	
% 7. Distraction (0 = no distraction, 1 = distraction)
% 8. Congruence (0 = congruent, 1 = incongruent)

clear v5 v6
v1=find(video_info(:,1)>1  & video_info(:,3)==2 & video_info(:,6)==2 & video_info(:,5)==1 & video_info(:,8)==0); % Yielding - Green or Cyan, 35 m, WALK
v2=find(video_info(:,1)==1 & video_info(:,3)==2 & video_info(:,6)==5 & video_info(:,5)==1 & video_info(:,8)==0); % Yielding - Red, 35 m, Bar
v3=find(video_info(:,1)>1  & video_info(:,3)==2 & video_info(:,6)==2 & video_info(:,5)==2 & video_info(:,8)==0); % Yielding - Green or Cyan, 50 m, WALK
v4=find(video_info(:,1)==1 & video_info(:,3)==2 & video_info(:,6)==5 & video_info(:,5)==2 & video_info(:,8)==0); % Yielding - Red, 50 m, Bar
figure;hold on;grid on;box on
title('Effect of activation distance (for egocentric messages and bar messages)')
calculate_means
plot(time,V1,':','Linewidth',5,'color',[0 255 0]/255);
plot(time,V2,':','Linewidth',5,'color',[255 2 0]/255);
plot(time,V3,'Linewidth',5,'color',[0 255 0]/255);
plot(time,V4,'Linewidth',5,'color',[255 2 0]/255);
xlabel('Time (s)')
ylabel('Percentage of participants pressing response key')
set(gca,'LooseInset',[0.01 0.01 0.01 0.01],'xlim',[0 12],'xtick',1:12,'ylim',[0 60])
legend(['Yielding - 35 m - Green & cyan - Egocentric (WALK) (' num2str(length(v1)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v1p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v1p,'omitnan')) '%)'],...
    ['Yielding - 35 m - Red - Bar (' num2str(length(v2)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v2p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v2p,'omitnan')) '%)'],...
    ['Yielding - 50 m - Green & cyan - Egocentric (WALK) (' num2str(length(v3)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v3p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v3p,'omitnan')) '%)'],...
    ['Yielding - 50 m - Red - Bar (' num2str(length(v4)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v4p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v4p,'omitnan')) '%)'],...
    'autoupdate','off');
h=findobj('FontName','Helvetica'); set(h,'FontSize',16,'Fontname','Arial')
rectangle('position',[0 0 1 100],'facecolor',[127 127 127 20]/255,'Linewidth',.5);
rectangle('position',[11 0 10 100],'facecolor',[127 127 127 20]/255,'Linewidth',.5);
[pyielding1,pyielding2]=deal(NaN(1,length(V)));
for i=1:length(V)
    [~,pyielding1(i)]=ttest(mean(mean(key_pressesf(:,v1,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v3,V(i):V(i)+9),3,'omitnan'),2,'omitnan')); % E (significance of difference between activation distances for green/cyan egocentric messages (WALK))
    [~,pyielding2(i)]=ttest(mean(mean(key_pressesf(:,v2,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v4,V(i):V(i)+9),3,'omitnan'),2,'omitnan')); % B (significance of difference between activation distances was observed for red bar messages)
    if pyielding1(i)<0.005 && pyielding2(i)<0.005 
        text(i-0.5,2,'EB','horizontalalignment','center','color','b')
    elseif pyielding1(i)<0.005 && pyielding2(i)>=0.005 
        text(i-0.5,2,'E','horizontalalignment','center','color','b')
     elseif pyielding1(i)>0.005 && pyielding2(i)<0.005 
        text(i-0.5,2,'B','horizontalalignment','center','color','b')
    elseif pyielding1(i)>0.005 && pyielding2(i)>=0.005 
        text(i-0.5,2,'','horizontalalignment','center','color','b')
    end
end
h=findobj('FontName','Helvetica'); set(h,'FontSize',16,'Fontname','Arial')
if savepng==1;set(gcf,'WindowState', 'maximized');print(gcf,'Figure9.png','-dpng','-r300');end
%% Distraction
% Figure 10. Percentage of participants pressing the response key as a function of elapsed video time, distinguishing between yielding behaviour and visual distraction. 

% 1. eHMI colour (1 = red, 2 = green, 3 = cyan)	
% 2. eHMI position (1 = bumper, 2 = windshield, 3 = roof)
% 3. AV behaviour (1 = non-yielding, 2 = yielding)
% 4. eHMI activation (1 = no, 2 = yes)	
% 5. eHMI activation distance (0 = no activation, 1 = 35 m, 2 = 50 m)
% 6. eHMI message (0 = no activation, 1 = DON'T WALK, 2 = WALK, 3 = WON'T STOP, 4 = WILL STOP, 5 = bar of light)	
% 7. Distraction (0 = no distraction, 1 = distraction)
% 8. Congruence (0 = congruent, 1 = incongruent)

clear v5 v6
v1=find(video_info(:,3)==2 & video_info(:,7)==1 & video_info(:,5)>0 & video_info(:,8)==0); % yielding, distraction
v2=find(video_info(:,3)==2 & video_info(:,7)==0 & video_info(:,5)>0 & video_info(:,8)==0); % yielding, no distraction
v3=find(video_info(:,3)==1 & video_info(:,7)==1 & video_info(:,5)>0 & video_info(:,8)==0); % non-yielding, distraction
v4=find(video_info(:,3)==1 & video_info(:,7)==0 & video_info(:,5)>0 & video_info(:,8)==0); % non-yielding, no distraction
figure;hold on;grid on;box on
title('Effect of distraction')
calculate_means
plot(time,V1,'-','Linewidth',5,'color',[0 0 0]);
plot(time,V2,'-','Linewidth',5,'color',[1 0 1]);
plot(time,V3,':','Linewidth',5,'color',[0 0 0]);
plot(time,V4,':','Linewidth',5,'color',[1 0 1]);
xlabel('Time (s)')
ylabel('Percentage of participants pressing response key')
set(gca,'LooseInset',[0.01 0.01 0.01 0.01],'xlim',[0 12],'xtick',1:12,'ylim',[0 60])
legend(['Yielding - Distraction (' num2str(length(v1)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v1p)) '%, \itSD\rm = ' sprintf('%0.1f',std(v1p)) '%)'],...
       ['Yielding - No distraction (' num2str(length(v2)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v2p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v2p,'omitnan')) '%)'],...
       ['Non-yielding - Distraction (' num2str(length(v3)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v3p)) '%, \itSD\rm = ' sprintf('%0.1f',std(v3p)) '%)'],...
       ['Non-yielding - No distraction (' num2str(length(v4)) ' videos, \itM\rm = ' sprintf('%0.1f',mean(v4p,'omitnan')) '%, \itSD\rm = ' sprintf('%0.1f',std(v4p,'omitnan')) '%)'],'autoupdate','off');
rectangle('position',[0 0 1 100],'facecolor',[127 127 127 20]/255,'Linewidth',.5);
rectangle('position',[11 0 10 100],'facecolor',[127 127 127 20]/255,'Linewidth',.5);
[pyielding,pnonyielding]=deal(NaN(1,length(V)));
for i=1:length(V)
    [~,pyielding(i)]=ttest(mean(mean(key_pressesf(:,v1,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v2,V(i):V(i)+9),3,'omitnan'),2,'omitnan')); % Y
    if pyielding(i)<0.005
        text(i-0.5,4,'Y','horizontalalignment','center','color','b')
    end
    [~,pnonyielding(i)]=ttest(mean(mean(key_pressesf(:,v3,V(i):V(i)+9),3,'omitnan'),2,'omitnan'),mean(mean(key_pressesf(:,v4,V(i):V(i)+9),3,'omitnan'),2,'omitnan')); % NY
    if pnonyielding(i)<0.005
        text(i-0.5,2,'NY','horizontalalignment','center','color','b')
    end
end
h=findobj('FontName','Helvetica'); set(h,'FontSize',16,'Fontname','Arial')
if savepng==1;set(gcf,'WindowState', 'maximized');print(gcf,'Figure10.png','-dpng','-r300');end
%% PerceivedSafety percentage computed as: time pressed / total time.
PerceivedSafety = NaN(size(key_pressesf,1),size(key_pressesf,2)); % Allocate space for Perceived Safety percentages
for nPar = 1 : size(key_pressesf,1)
    for nVid = 1 :  size(key_pressesf,2)
        PerceivedSafety(nPar,nVid) = 100*mean(key_pressesf(nPar,nVid,51:110)); % Only use the time between 5 and 11 seconds
    end
end
PerceivedSafetyMean = mean(PerceivedSafety,'omitnan');
PerceivedSafetyMean = PerceivedSafetyMean';
%% Supplementary table with perceived safety percentages per video
videos_PerceivedSafety = num2cell([NaN(length(PerceivedSafetyMean),1) video_info PerceivedSafetyMean NaN(length(PerceivedSafetyMean),1)]);

for i = 1:size(videos_PerceivedSafety,1) % loop over 276 videos
    videos_PerceivedSafety{i,1} = ['video_' num2str(i)]; % video id
    
    % eHMI colour
    if videos_PerceivedSafety{i,2} == 1
        videos_PerceivedSafety{i, 11} = 'red -';
        videos_PerceivedSafety{i, 2} = 'red';
    elseif videos_PerceivedSafety{i,2} == 2
        videos_PerceivedSafety{i, 11} = 'green -';
        videos_PerceivedSafety{i, 2} = 'green';
    elseif videos_PerceivedSafety{i,2} == 3
        videos_PerceivedSafety{i, 11} = 'cyan -';
        videos_PerceivedSafety{i, 2} = 'cyan';
    elseif videos_PerceivedSafety{i,2} == 0
        videos_PerceivedSafety{i, 11} = '';
        videos_PerceivedSafety{i, 2} = 'none';
    end
    
    if videos_PerceivedSafety{i,2} ~= 0 % if a colour then add space
        videos_PerceivedSafety{i, 11} = strcat(videos_PerceivedSafety{i, 11}, ' ');
    end
    
    % eHMI position
    if videos_PerceivedSafety{i,3} == 1
        videos_PerceivedSafety{i, 11} = strcat(videos_PerceivedSafety{i, 11}, 'bumper -');
        videos_PerceivedSafety{i, 3} = 'bumper';
    elseif videos_PerceivedSafety{i,3} == 2
        videos_PerceivedSafety{i, 11} = strcat(videos_PerceivedSafety{i, 11}, 'windshield -');
        videos_PerceivedSafety{i, 3} = 'windshield';
    elseif videos_PerceivedSafety{i,3} == 3
        videos_PerceivedSafety{i, 11} = strcat(videos_PerceivedSafety{i, 11}, 'roof -');
        videos_PerceivedSafety{i, 3} = 'roof';
    end
    
    % AV behaviour
    if videos_PerceivedSafety{i,4} == 1
        videos_PerceivedSafety{i, 11} = strcat(videos_PerceivedSafety{i, 11}, ' non-yielding -');
        videos_PerceivedSafety{i, 4} = 'non-yielding';
    elseif videos_PerceivedSafety{i,4} == 2
        videos_PerceivedSafety{i, 11} = strcat(videos_PerceivedSafety{i, 11}, ' yielding -');
        videos_PerceivedSafety{i, 4} = 'yielding';
    end
    
    % eHMI activation
    if videos_PerceivedSafety{i,5} == 1
        videos_PerceivedSafety{i, 11} = strcat(videos_PerceivedSafety{i, 11}, ' display inactive -');
        videos_PerceivedSafety{i, 5} = 'inactive';
    elseif videos_PerceivedSafety{i,5} == 2
        videos_PerceivedSafety{i, 11} = strcat(videos_PerceivedSafety{i, 11}, ' display active -');
        videos_PerceivedSafety{i, 5} = 'active';
    end
    
    % eHMI activation distance
    if videos_PerceivedSafety{i,6} == 0
        videos_PerceivedSafety{i, 11} = strcat(videos_PerceivedSafety{i, 11}, ' no activation -');
        videos_PerceivedSafety{i, 6} = 'none';
    elseif videos_PerceivedSafety{i,6} == 1
        videos_PerceivedSafety{i, 11} = strcat(videos_PerceivedSafety{i, 11}, ' activation at 35 m -');
        videos_PerceivedSafety{i, 6} = '35 m';
    elseif videos_PerceivedSafety{i,6} == 2
        videos_PerceivedSafety{i, 11} = strcat(videos_PerceivedSafety{i, 11}, ' activation at 50 m -');
        videos_PerceivedSafety{i, 6} = '50 m';
    end
    
    % eHMI message
    if videos_PerceivedSafety{i,7} == 0
        videos_PerceivedSafety{i, 11} = strcat(videos_PerceivedSafety{i, 11}, ' none -');
        videos_PerceivedSafety{i, 7} = 'none';
    elseif videos_PerceivedSafety{i,7} == 1
        videos_PerceivedSafety{i, 11} = strcat(videos_PerceivedSafety{i, 11}, ' dont walk -');
        videos_PerceivedSafety{i, 7} = 'dont walk';
    elseif videos_PerceivedSafety{i,7} == 2
        videos_PerceivedSafety{i, 11} = strcat(videos_PerceivedSafety{i, 11}, ' walk -');
        videos_PerceivedSafety{i, 7} = 'walk';
    elseif videos_PerceivedSafety{i,7} == 3
        videos_PerceivedSafety{i, 11} = strcat(videos_PerceivedSafety{i, 11}, ' wont stop -');
        videos_PerceivedSafety{i, 7} = 'wont stop';
    elseif videos_PerceivedSafety{i,7} == 4
        videos_PerceivedSafety{i, 11} = strcat(videos_PerceivedSafety{i, 11}, ' will stop -');
        videos_PerceivedSafety{i, 7} = 'will stop';
    elseif videos_PerceivedSafety{i,7} == 5
        videos_PerceivedSafety{i, 11} = strcat(videos_PerceivedSafety{i, 11}, ' bar -');
        videos_PerceivedSafety{i, 7} = 'bar';
    end
    
    % distraction
    if videos_PerceivedSafety{i,8} == 0
        videos_PerceivedSafety{i, 11} = strcat(videos_PerceivedSafety{i, 11}, ' without distraction');
        videos_PerceivedSafety{i, 8} = 'none';
    elseif videos_PerceivedSafety{i,8} == 1
        videos_PerceivedSafety{i, 11} = strcat(videos_PerceivedSafety{i, 11}, ' with distraction');
        videos_PerceivedSafety{i, 8} = 'distraction';
    end

    % congruence
    if videos_PerceivedSafety{i,9} == 0
        videos_PerceivedSafety{i, 9} = 'congruent';
    elseif videos_PerceivedSafety{i,9} == 1
        videos_PerceivedSafety{i, 9} = 'incongruent';
    end
end

videos_PerceivedSafety_sorted_nonyielding = sortrows(videos_PerceivedSafety(video_info(:,3)==1,:), 10); % non-yielding
videos_PerceivedSafety_sorted_yielding = sortrows(videos_PerceivedSafety(video_info(:,3)==2,:), 10); % yielding
